/**
 * Smartideo 古腾堡编辑器块 JavaScript
 */
(function( wp ) {
    var __ = wp.i18n.__;
    var el = wp.element.createElement;
    var registerBlockType = wp.blocks.registerBlockType;
    var TextControl = wp.components.TextControl;
    var Placeholder = wp.components.Placeholder;
    var InspectorControls = wp.editor.InspectorControls;
    var PanelBody = wp.components.PanelBody;
    var TextareaControl = wp.components.TextareaControl;
    var Spinner = wp.components.Spinner;
    
    // 视频平台正则表达式匹配模式
    var videoPatterns = {
        youtube: { 
            pattern: /https?:\/\/(?:www\.)?(?:youtube\.com\/(?:watch\?(?:.*?&)?v=|embed\/|v\/|live\/)|youtu\.be\/)([a-zA-Z0-9_-]{11})/i, 
            embedUrl: function(match) {
                return 'https://www.youtube.com/embed/' + match[1];
            }
        },
        bilibili: { 
            pattern: /https?:\/\/(?:www\.|mobile\.)?bilibili\.(?:com|tv)\/(?:video\/(BV|av)([a-zA-Z0-9_\-]+)|BV([a-zA-Z0-9_\-]+)|av(\d+))/i, 
            embedUrl: function(match) {
                var type = match[1] || (match[3] ? 'BV' : 'av');
                var id = match[2] || match[3] || match[4];
                
                if (type === 'BV') {
                    return 'https://player.bilibili.com/player.html?bvid=' + (id.startsWith('BV') ? id : 'BV' + id);
                } else {
                    return 'https://player.bilibili.com/player.html?aid=' + id;
                }
            }
        },
        youku: { 
            pattern: /https?:\/\/(?:v|vo|u)\.youku\.com\/(?:v_show\/id_|video\/id_)?([a-z0-9_\-]+)(?:\.html|\.htm)?/i, 
            embedUrl: function(match) {
                return 'https://player.youku.com/embed/' + match[1];
            }
        },
        qq: { 
            pattern: /https?:\/\/(?:v|vm)\.qq\.com\/(?:[a-z0-9_\.\/]+(?:\?|#|&)vid=([a-z0-9_\-]+)|(?:[a-z0-9_\.\/]+\/|x\/page\/)([a-z0-9_\-]+))/i, 
            embedUrl: function(match) {
                var vid = match[1] || match[2];
                return 'https://v.qq.com/iframe/player.html?vid=' + vid + '&auto=0&tiny=0&auto=0';
            }
        },
        douyin: { 
            pattern: /https?:\/\/(?:www\.)?douyin\.com\/(?:video\/|jingxuan\?modal_id=|share\/?(?:video\/)?|user\d+\?modal_id=)(\d+)/i, 
            embedUrl: function(match) {
                return 'https://open.douyin.com/player/video?vid=' + match[1] + '&autoplay=0';
            }
        },
        bilibili_audio: { 
            pattern: /https?:\/\/(?:www\.)?bilibili\.com\/audio\/au(\d+)/i, 
            embedUrl: function(match) {
                return 'https://player.bilibili.com/player.html?aid=' + match[1];
            }
        },
        music163: { 
            pattern: /https?:\/\/(?:music\.163\.com|y\.163\.com)(?:\/\#|\/|)?\/(?:song|song\?id=|program\?id=|mv\?id=)(\d+)/i, 
            embedUrl: function(match) {
                return 'https://music.163.com/outchain/player?type=2&id=' + match[1] + '&auto=0&height=90';
            }
        },
        musicqq: { 
            pattern: /https?:\/\/(?:y\.qq\.com|c\.y\.qq\.com|i\.y\.qq\.com)\/(?:n\/ryqq\/songDetail\/|n\/yqq\/song\/|playsong\.html\?songid=|new\.music\.qq\.com\/v8\/fcg-bin\/play\.fcg\?songmid=)([a-zA-Z0-9_=-]+)/i, 
            embedUrl: function(match) {
                return 'https://i.y.qq.com/n2/m/outchain/player/index.html?songid=' + match[1] + '&songtype=0';
            }
        }
    };
    
    // 获取嵌入URL和平台信息
    function getEmbedUrl(url) {
        for (var platform in videoPatterns) {
            if (videoPatterns.hasOwnProperty(platform)) {
                var match = url.match(videoPatterns[platform].pattern);
                if (match) {
                    var embedUrl = videoPatterns[platform].embedUrl(match);
                    return {
                        url: embedUrl,
                        platform: platform,
                        requiresJsonProcessing: videoPatterns[platform].requiresJsonProcessing || false
                    };
                }
            }
        }
        return null;
    }
    
    // 注册块
    registerBlockType( 'smartideo/video', {
        title: __( 'Smartideo 视频', 'smartideo' ),
        icon: 'format-video',
        category: 'common',
        keywords: [
            __( '视频', 'smartideo' ),
            __( 'YouTube', 'smartideo' ),
            __( '哔哩哔哩', 'smartideo' ),
            __( '腾讯视频', 'smartideo' ),
            __( '优酷', 'smartideo' )
        ],
        attributes: {
            url: {
                type: 'string',
                default: ''
            },
            description: {
                type: 'string',
                default: ''
            }
        },
        
        // 编辑界面
        edit: function( props ) {
            var attributes = props.attributes;
            var setAttributes = props.setAttributes;
            var [isLoading, setIsLoading] = wp.element.useState(false);
            var [error, setError] = wp.element.useState('');
            
            var onChangeUrl = function( value ) {
                setAttributes( { url: value } );
                setError('');
            };
            
            var onChangeDescription = function( value ) {
                setAttributes( { description: value } );
            };
            
            var renderPlaceholder = function() {
                return el( Placeholder, {
                    icon: 'format-video',
                    label: __( 'Smartideo 视频', 'smartideo' ),
                    instructions: __( '请输入视频或音乐网址，支持YouTube、哔哩哔哩、抖音、优酷、腾讯视频、网易云音乐、QQ音乐等平台', 'smartideo' )
                },
                el( TextControl, {
                    label: __( '视频URL', 'smartideo' ),
                    value: attributes.url,
                    onChange: onChangeUrl,
                    placeholder: __( '请输入视频URL', 'smartideo' ),
                    style: {
                        width: '100%',
                        display: 'block',
                        fontSize: '14px',
                        minHeight: '40px',
                        boxSizing: 'border-box',
                        margin: '0',
                        padding: '8px 12px'
                    },
                    className: 'wide-input-control'
                })
                );
            };
            
            var renderVideoPreview = function() {
                var embedInfo = getEmbedUrl(attributes.url);
                
                if (!embedInfo) {
                    return el('div', {
                        className: 'smartideo-gutenberg-error',
                        style: {
                            padding: '20px',
                            backgroundColor: '#fff',
                            border: '1px solid #ff9800',
                            borderRadius: '4px',
                            color: '#e65100',
                            textAlign: 'center'
                        }
                    },
                    __( '无法识别的视频链接，请检查URL格式是否正确', 'smartideo' ));
                }
                
                var isMusic = attributes.url.includes('music.163.com') || attributes.url.includes('y.qq.com');
                var isDouyin = embedInfo.platform === 'douyin';
                var smartideoSettings = wp.data.select('core/editor').getEditedPostAttribute('meta')['smartideo_settings'] || {};
                var isResponsive = smartideoSettings.response === '1';
                
                var height = isResponsive ? 'auto' : (isMusic ? '60px' : (isDouyin ? '500px' : '400px'));
                
                if (embedInfo.requiresJsonProcessing) {
                    var container = el('div', {
                        className: 'smartideo-gutenberg-json-notice',
                        style: {
                            position: 'relative',
                            width: '100%',
                            height: height,
                            backgroundColor: '#f5f5f5',
                            borderRadius: '4px',
                            display: 'flex',
                            justifyContent: 'center',
                            alignItems: 'center',
                            border: '1px dashed #9e9e9e',
                            color: '#616161'
                        }
                    },
                    el('span', null, embedInfo.platform === 'douyin' ? 
                        '抖音视频将在前台正确显示' : '视频将在前台正确显示')
                    );
                    
                    if (attributes.description) {
                        return el('div', {
                            className: 'smartideo-gutenberg-preview-with-description'
                        },
                        container,
                        el('div', {
                            className: 'smartideo-gutenberg-description',
                            style: {
                                marginTop: '10px',
                                padding: '10px',
                                backgroundColor: '#fff',
                                border: '1px solid #e0e0e0',
                                borderRadius: '4px',
                                fontSize: '14px',
                                lineHeight: '1.5',
                                color: '#333'
                            }
                        }, attributes.description)
                        );
                    }
                    
                    return container;
                }
                
                var videoContainer = el('div', {
                    className: 'smartideo-gutenberg-video-preview smartideo-responsive-preview',
                    style: {
                        position: 'relative',
                        width: '100%',
                        height: isMusic ? height : 'auto',
                        backgroundColor: '#000',
                        borderRadius: '4px',
                        overflow: 'hidden'
                    }
                },
                el('iframe', {
                    src: embedInfo.url,
                    width: '100%',
                    height: '100%',
                    frameBorder: '0',
                    allowFullScreen: true,
                    title: __( '视频预览', 'smartideo' )
                }));
                
                if (attributes.description) {
                    return el('div', {
                        className: 'smartideo-gutenberg-preview-with-description'
                    },
                    videoContainer,
                    el('div', {
                        className: 'smartideo-gutenberg-description',
                        style: {
                            marginTop: '10px',
                            padding: '10px',
                            backgroundColor: '#fff',
                            border: '1px solid #e0e0e0',
                            borderRadius: '4px',
                            fontSize: '14px',
                            lineHeight: '1.5',
                            color: '#333'
                        }
                    }, attributes.description)
                    );
                }
                
                return videoContainer;
            };
            
            var renderWithUrl = function() {
                return el( 'div', {
                    className: 'smartideo-gutenberg-edit',
                },
                el( InspectorControls, {},
                    el( PanelBody, {
                        title: __( '视频设置', 'smartideo' ),
                        initialOpen: true
                    },
                    el( TextareaControl, {
                        label: __( '视频URL', 'smartideo' ),
                        value: attributes.url,
                        onChange: onChangeUrl,
                        rows: 2,
                        placeholder: __( '请输入视频URL', 'smartideo' ),
                        style: {
                            width: '100%',
                            display: 'block',
                            fontSize: '14px',
                            boxSizing: 'border-box'
                        }
                    }),
                    el( TextareaControl, {
                        label: __( '视频描述 (选填)', 'smartideo' ),
                        value: attributes.description,
                        onChange: onChangeDescription,
                        rows: 3,
                        placeholder: __( '输入视频的描述文本，将显示在视频下方', 'smartideo' )
                    })
                    )
                ),
                el( 'div', {
                    className: 'smartideo-gutenberg-preview-container',
                    style: {
                        padding: '15px',
                        backgroundColor: '#f9f9f9',
                        borderRadius: '4px',
                        border: '1px solid #e0e0e0'
                    }
                },
                error ? 
                    el('div', {
                        style: {
                            color: '#d32f2f',
                            padding: '10px',
                            backgroundColor: '#ffebee',
                            borderRadius: '4px',
                            marginBottom: '10px'
                        }
                    }, error) : 
                    null,
                isLoading ? 
                    el('div', {
                        style: {
                            display: 'flex',
                            justifyContent: 'center',
                            alignItems: 'center',
                            height: '200px'
                        }
                    }, el(Spinner)) : 
                    renderVideoPreview()
                )
                );
            };
            
            return attributes.url ? renderWithUrl() : renderPlaceholder();
        },
        
        // 保存函数
        save: function( props ) {
            return el('div', {
                className: 'smartideo-block',
                'data-url': props.attributes.url,
                'data-description': props.attributes.description
            });
        }
    });
})( window.wp );