<?php
/**
 * Smartideo 古腾堡编辑器支持
 * @package Smartideo
 */

/** 注册古腾堡块 */
function smartideo_register_gutenberg_block() {
    // 检查是否支持古腾堡
    if ( ! function_exists( 'register_block_type' ) ) {
        return;
    }

    // 注册块的JavaScript
    wp_register_script(
        'smartideo-gutenberg',
        SMARTIDEO_URL . '/static/smartideo-gutenberg.js',
        array( 'wp-blocks', 'wp-i18n', 'wp-element', 'wp-editor' ),
        SMARTIDEO_VERSION,
        true
    );

    // 注册块的样式
    wp_register_style(
        'smartideo-gutenberg-editor',
        SMARTIDEO_URL . '/static/smartideo-gutenberg.css',
        array( 'wp-edit-blocks' ),
        SMARTIDEO_VERSION
    );

    // 注册前端样式（使用已有的CSS）
    wp_register_style(
        'smartideo-gutenberg-frontend',
        SMARTIDEO_URL . '/static/smartideo.css',
        array(),
        SMARTIDEO_VERSION
    );

    // 注册块
    register_block_type(
        'smartideo/video',
        array(
            'editor_script' => 'smartideo-gutenberg',
            'editor_style'  => 'smartideo-gutenberg-editor',
            'style'         => 'smartideo-gutenberg-frontend',
            'render_callback' => 'smartideo_render_gutenberg_block',
            'attributes' => array(
                'url' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'description' => array(
                    'type' => 'string',
                    'default' => '',
                ),
            ),
        )
    );
}

/** 渲染古腾堡块 */
function smartideo_render_gutenberg_block( $attributes ) {
    global $smartideo;
    
    // 检查URL和Smartideo实例是否可用
    if ( empty( $attributes['url'] ) || ! $smartideo ) {
        return '';
    }
    
    $url = esc_url( $attributes['url'] );
    $description = isset( $attributes['description'] ) ? $attributes['description'] : '';
    
    // 尝试直接使用Smartideo的短代码处理功能
    // 检查Smartideo是否有parse_shortcode或类似方法
    if ( method_exists( $smartideo, 'parse_shortcode' ) ) {
        $html = $smartideo->parse_shortcode( array( 'url' => $url ) );
    }
    // 或者检查是否有process_shortcode方法
    else if ( method_exists( $smartideo, 'process_shortcode' ) ) {
        $html = $smartideo->process_shortcode( array( 'url' => $url ) );
    }
    // 或者检查是否有handle_shortcode方法
    else if ( method_exists( $smartideo, 'handle_shortcode' ) ) {
        $html = $smartideo->handle_shortcode( array( 'url' => $url ) );
    }
    // 如果以上方法都不存在，尝试使用现有的嵌入处理程序
    else {
            // 视频平台正则匹配模式
        $platforms = array(
            'youtube' => '~https?://(?:www\.)?(?:youtube\.com/(?:watch\?(?:.*?&)?v=|embed/|v/|live/)|youtu\.be/)([a-zA-Z0-9_-]{11})~i',
            'bilibili' => '~https?://(?:www\.|mobile\.)?bilibili\.(?:com|tv)/(?:video/(BV|av)([a-zA-Z0-9_=-]+)|BV([a-zA-Z0-9_=-]+)|av(\d+))~i',
            'youku' => '~https?://(?:v|vo|u)\.youku\.com/(?:v_show/id_|video/id_)?([a-z0-9_=-]+)(?:\.html|\.htm)?~i',
            'qq' => '~https?://(?:v|vm)\.qq\.com/(?:[a-z0-9_\./]+(?:\?|#|&)vid=([a-z0-9_=-]+)|(?:[a-z0-9_\./]+/|x/page/)([a-z0-9_=-]+))~i',
            'douyin' => '~https?://(?:www\.)?douyin\.com/(?:video/|jingxuan\?modal_id=|share/?(?:video/)?|user\d+\?modal_id=)(\d+)~i',
            'bilibili_audio' => '~https?://(?:www\.)?bilibili\.com/audio/au(\d+)~i',
            'music163' => '~https?://(?:music\.163\.com|y\.163\.com)(?:/\#|/|)?/(?:song|song\?id=|program\?id=|mv\?id=)(\d+)~i',
            'musicqq' => '~https?://(?:y\.qq\.com|c\.y\.qq\.com|i\.y\.qq\.com)/(?:n/ryqq/songDetail/|n/yqq/song/|playsong\.html\?songid=|new\.music\.qq\.com/v8/fcg-bin/play\.fcg\?songmid=)([a-zA-Z0-9_=-]+)~i'
        );
        
        foreach ( $platforms as $platform => $pattern ) {
            if ( preg_match( $pattern, $url, $matches ) ) {
                // 处理不同平台的匹配结果
                $processed_matches = $matches;
                
                // 为Bilibili处理多个捕获组
                if ( $platform === 'bilibili' ) {
                    // 找到非空的视频ID捕获组
                    for ($i = 2; $i <= 4; $i++) {
                        if ( !empty($matches[$i]) ) {
                            $processed_matches['video_id'] = $matches[$i];
                            break;
                        }
                    }
                }
                // 为腾讯视频处理多个捕获组
                else if ( $platform === 'qq' ) {
                    // 优先使用第一个捕获组，如果为空则使用第二个
                    $processed_matches['video_id'] = !empty($matches[1]) ? $matches[1] : $matches[2];
                }
                // 为其他平台设置video_id为第一个捕获组
                else if ( isset($matches[1]) ) {
                    $processed_matches['video_id'] = $matches[1];
                }
                
                $method = "smartideo_embed_handler_{$platform}";
                if ( method_exists( $smartideo, $method ) ) {
                    // 调用对应的处理方法
                    $html = call_user_func_array( array( $smartideo, $method ), array( $processed_matches, array(), $url, array() ) );
                    // 确保返回的是HTML而非URL文本
                    if ( $html && $html != $url && strpos( $html, '<iframe' ) !== false ) {
                        break;
                    }
                }
            }
        }
        
        // 如果没有找到合适的处理方法，尝试手动生成嵌入HTML
        if ( empty( $html ) ) {
            $html = smartideo_manual_embed( $url );
        }
    }
    
    // 如果有描述，添加到视频下方
    if ( !empty( $description ) ) {
        // 添加额外的检查确保描述不为空
        $clean_description = trim( $description );
        if ( !empty( $clean_description ) ) {
            $html .= '<div class="smartideo-gutenberg-description">' . wp_kses_post( $clean_description ) . '</div>';
        }
    }
    
    return $html;
}

/** 处理抖音视频嵌入 */
function smartideo_process_douyin_video( $matches ) {
    $video_id = $matches[1];
    $api_url = 'https://open.douyin.com/api/douyin/v1/video/get_iframe_by_video?video_id=' . $video_id;
    
    // 尝试获取API响应
    $response = wp_remote_get( $api_url, array(
        'timeout' => 10,
        'headers' => array(
            'Accept' => 'application/json',
        ),
    ));
    
    // 检查响应是否成功
    if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        // 检查JSON解析是否成功，并且是否包含iframe_code
        if ( $data && isset( $data['data']['iframe_code'] ) ) {
            // 直接返回iframe代码
            return $data['data']['iframe_code'];
        }
    }
    
    // 如果API调用失败，返回一个备用的iframe嵌入方式
    return '<iframe width="560" height="315" src="https://open.douyin.com/player/video?vid=' . $video_id . '&autoplay=0" frameborder="0" allowfullscreen></iframe>';
}

/** 手动生成视频嵌入HTML */
function smartideo_manual_embed( $url ) {
    // 视频平台匹配模式和嵌入配置
    $video_patterns = array(
        'youtube' => array(
            'pattern' => '~https?://(?:www\.)?(?:youtube\.com/(?:watch\?(?:.*?&)?v=|embed/|v/|live/)|youtu\.be/)([a-zA-Z0-9_-]{11})~i',
            'embed' => 'https://www.youtube.com/embed/$1'
        ),
        'bilibili' => array(
            'pattern' => '~https?://(?:www\.|mobile\.)?bilibili\.(?:com|tv)/(?:video/(BV|av)([a-zA-Z0-9_=-]+)|BV([a-zA-Z0-9_=-]+)|av(\d+))~i',
            'callback' => 'smartideo_process_bilibili_video'
        ),
        'youku' => array(
            'pattern' => '~https?://(?:v|vo|u)\.youku\.com/(?:v_show/id_|video/id_)?([a-z0-9_=-]+)(?:\.html|\.htm)?~i',
            'embed' => 'https://player.youku.com/embed/$1'
        ),
        'qq' => array(
            'pattern' => '~https?://(?:v|vm)\.qq\.com/(?:[a-z0-9_\./]+(?:\?|#|&)vid=([a-z0-9_=-]+)|(?:[a-z0-9_\./]+/|x/page/)([a-z0-9_=-]+))~i',
            'callback' => 'smartideo_process_qq_video'
        ),
        'douyin' => array(
            'pattern' => '~https?://(?:www\.)?douyin\.com/(?:video/|jingxuan\?modal_id=|share/?(?:video/)?|user\d+\?modal_id=)(\d+)~i',
            'callback' => 'smartideo_process_douyin_video'
        ),
        'bilibili_audio' => array(
            'pattern' => '~https?://(?:www\.)?bilibili\.com/audio/au(\d+)~i',
            'embed' => 'https://player.bilibili.com/player.html?aid=$1'
        ),
        'music163' => array(
            'pattern' => '~https?://(?:music\.163\.com|y\.163\.com)(?:/\#|/|)?/(?:song|song\?id=|program\?id=|mv\?id=)(\d+)~i',
            'embed' => 'https://music.163.com/outchain/player?type=2&id=$1&auto=0&height=90'
        ),
        'musicqq' => array(
            'pattern' => '~https?://(?:y\.qq\.com|c\.y\.qq\.com|i\.y\.qq\.com)/(?:n/ryqq/songDetail/|n/yqq/song/|playsong\.html\?songid=|new\.music\.qq\.com/v8/fcg-bin/play\.fcg\?songmid=)([a-zA-Z0-9_=-]+)~i',
            'embed' => 'https://i.y.qq.com/n2/m/outchain/player/index.html?songid=$1&songtype=0'
        )
    );
    
    // 检查URL是否匹配支持的平台
    foreach ( $video_patterns as $platform => $config ) {
        if ( preg_match( $config['pattern'], $url, $matches ) ) {
            if ( isset( $config['callback'] ) && function_exists( $config['callback'] ) ) {
                $embed_url = call_user_func( $config['callback'], $matches );
            } else {
                $embed_url = $config['embed'];
                // 替换嵌入URL中的占位符
                for ( $i = 1; $i < count( $matches ); $i++ ) {
                    $embed_url = str_replace( '$' . $i, $matches[$i], $embed_url );
                }
            }
            
            // 判断是否是音乐类型视频
            $is_music = strpos( $url, 'music.163.com' ) !== false || 
                        strpos( $url, 'y.qq.com' ) !== false;
            
            $height = $is_music ? '60' : '400';
            
            // 获取Smartideo设置
            $options = get_option( 'smartideo_settings', array() );
            
            // 构建视频容器类名
            $container_class = 'smartideo-container';
            // 保持与PHP主文件一致的响应式选项命名
            if ( isset( $options['response'] ) && $options['response'] == '1' ) {
                $container_class .= ' smartideo-responsive';
            }
            
            // 检查是否已经是完整的iframe代码（如抖音返回的）
            if ( strpos( $embed_url, '<iframe' ) !== false ) {
                // 对于抖音等已返回完整iframe的情况，直接使用
                $html = '<div class="' . $container_class . '" style="max-width:100%;">';
                $html .= $embed_url;
            } else {
                // 普通情况，生成标准iframe
                $html = '<div class="' . $container_class . '" style="max-width:100%;">';
                $html .= '<iframe ';
                $html .= 'width="100%" ';
                $html .= 'height="' . $height . '" ';
                $html .= 'src="' . esc_url( $embed_url ) . '" ';
                $html .= 'frameborder="0" ';
                $html .= 'allowfullscreen ';
                $html .= 'allow="accelerometer; clipboard-write; encrypted-media; gyroscope; picture-in-picture" ';
                $html .= '></iframe>';
            }
            $html .= '</div>';
            
            return $html;
        }
    }
    
    // 如果无法识别URL，返回一个提示
    return '<div class="smartideo-error" style="padding:20px;background:#ffebee;border:1px solid #ffcdd2;color:#c62828;text-align:center;">
        无法解析视频URL，请确保这是支持的视频平台链接
    </div>';
}

/** 处理哔哩哔哩视频URL */
function smartideo_process_bilibili_video( $matches ) {
    // 提取视频ID和类型，支持多种匹配模式
    $type = '';
    $id = '';
    
    // 模式1: video/BV1xx411c7mX 或 video/av12345678
    if (isset($matches[2]) && !empty($matches[2])) {
        $type = strtoupper($matches[1]);
        $id = $matches[2];
    }
    // 模式2: BV1xx411c7mX (直接以BV开头的URL)
    else if (isset($matches[3]) && !empty($matches[3])) {
        $type = 'BV';
        $id = $matches[3];
    }
    // 模式3: av12345678 (直接以av开头的URL)
    else if (isset($matches[4]) && !empty($matches[4])) {
        $type = 'av';
        $id = $matches[4];
    }
    // 兼容旧格式
    else if (isset($matches[1]) && !empty($matches[1])) {
        $id = $matches[1];
        // 判断ID类型
        if (strtolower(substr($id, 0, 1)) === 'v') {
            $type = 'BV';
            $id = substr($id, 1); // 去掉v前缀
        } else {
            $type = 'av';
        }
    }
    
    // 构建嵌入URL
    if ($type === 'BV') {
        // 确保BV号格式正确（BV前缀+10-12个字符）
        if (!strtoupper(substr($id, 0, 2)) === 'BV') {
            $id = 'BV' . $id;
        }
        return 'https://player.bilibili.com/player.html?bvid=' . $id . '&autoplay=0';
    } else {
        // AV号格式
        return 'https://player.bilibili.com/player.html?aid=' . $id . '&autoplay=0';
    }
}

/** 处理QQ视频URL */
function smartideo_process_qq_video( $matches ) {
    // 提取视频ID，支持多种匹配模式
    $vid = '';
    
    // 模式1: 直接通过vid参数获取
    if (isset($matches[1]) && !empty($matches[1])) {
        $vid = $matches[1];
    }
    // 模式2: 通过URL路径获取
    else if (isset($matches[2]) && !empty($matches[2])) {
        $vid = $matches[2];
    }
    
    // 构建最新的QQ视频嵌入URL
    // 添加更多参数以支持更好的播放器体验
    return 'https://v.qq.com/iframe/player.html?vid=' . $vid . '&tiny=0&auto=0&loop=0&byline=0&portrait=0&share=0';
}

/** 加载古腾堡块 */
function smartideo_load_gutenberg_block() {
    // 在wp_loaded钩子上注册块
    add_action( 'init', 'smartideo_register_gutenberg_block' );
}

// 启动古腾堡支持
smartideo_load_gutenberg_block();